import textwrap

import pytest

from debputy.lsp.debputy_ls import DebputyLanguageServer
from debputy.lsp.languages.lsp_debian_watch import _debian_watch_hover
from debputy.lsprotocol.types import (
    TextDocumentIdentifier,
    HoverParams,
    MarkupContent,
)

try:
    from debputy.lsp.languages.lsp_debian_control import (
        _debian_control_completions,
        _debian_control_hover,
        _debian_control_semantic_tokens_full,
    )

    from pygls.server import LanguageServer
except ImportError:
    pass
from lsp_tests.lsp_tutil import (
    put_doc_with_cursor,
)


@pytest.mark.parametrize(
    "variant",
    [
        "foo-<CURSOR>@ANY_VERSION@",
        "foo-@<CURSOR>ANY_VERSION@",
        "foo-@ANY_VERSION<CURSOR>@",
    ],
)
def test_debian_watch_hover_doc_variables_matching(
    ls: "DebputyLanguageServer", variant: str
) -> None:
    dwatch_uri = "file:///nowhere/debian/watch"
    cursor_pos = put_doc_with_cursor(
        ls,
        dwatch_uri,
        "debian/watch",
        textwrap.dedent(
            f"""\
        Version: 5

        Source: https://example.org
        Matching-Pattern: {variant}
"""
        ),
    )

    hover_doc = _debian_watch_hover(
        ls,
        HoverParams(TextDocumentIdentifier(dwatch_uri), cursor_pos),
    )
    assert hover_doc is not None and isinstance(hover_doc.contents, MarkupContent)
    assert hover_doc.contents.value.startswith("# Variable `@ANY_VERSION@`")


@pytest.mark.parametrize(
    "variant",
    [
        "foo<CURSOR>-@ANY_VERSION@",
        "foo-@ANY_VERSION@<CURSOR>",
    ],
)
def test_debian_watch_hover_doc_variables_non_matching(
    ls: "DebputyLanguageServer", variant: str
) -> None:
    dwatch_uri = "file:///nowhere/debian/watch"
    cursor_pos = put_doc_with_cursor(
        ls,
        dwatch_uri,
        "debian/watch",
        textwrap.dedent(
            f"""\
                Version: 5

                Source: https://example.org
                Matching-Pattern: {variant}
        """
        ),
    )

    hover_doc = _debian_watch_hover(
        ls,
        HoverParams(TextDocumentIdentifier(dwatch_uri), cursor_pos),
    )
    provided_doc = ""
    if hover_doc is not None and isinstance(hover_doc.contents, MarkupContent):
        provided_doc = hover_doc.contents.value
    assert not provided_doc.startswith("# Variable `@ANY_VERSION@`")
